<?php

namespace App\Http\Controllers;

use App\Models\Booking;
use App\Models\BookingPayment;
use App\Models\Member;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Http;
use Tymon\JWTAuth\Facades\JWTAuth;
use App\Services\SafeHavenService;
use Carbon\Carbon;
use Illuminate\Support\Facades\Hash;
use App\Notifications\SendNewPasswordNotification;

use Firebase\JWT\JWT;
use Firebase\JWT\JWK;

class MemberController extends Controller
{



    /**
     * Get a JWT via given credentials.
     *
     * @return \Illuminate\Http\JsonResponse
     */
    public function login()
    {
        $credentials = request(['member_email', 'password']);

        if (! $token = auth('member')->attempt($credentials)) {
            return response()->json(['message' => 'Invalid login details'], 401);
        }

        return $this->respondWithToken($token);
    }

    /**
     * Get the authenticated User.
     *
     * @return \Illuminate\Http\JsonResponse
     */
    public function me()
    {
        return response()->json(auth('member')->user());
    }

    /**
     * Log the user out (Invalidate the token).
     *
     * @return \Illuminate\Http\JsonResponse
     */
    public function logout()
    {
        auth('member')->logout();

        return response()->json(['message' => 'Successfully logged out']);
    }

    /**
     * Refresh a token.
     *
     * @return \Illuminate\Http\JsonResponse
     */
    public function refresh()
    {
        return $this->respondWithToken(auth('member')->refresh());
    }

    /**
     * Get the token array structure.
     *
     * @param  string $token
     *
     * @return \Illuminate\Http\JsonResponse
     */
    protected function respondWithToken($token)
    {
        return response()->json([
            'token' => $token,
            'token_type' => 'bearer',
            'member' => auth('member')->user(),
            'expires_in' => auth('member')->factory()->getTTL() * 60
        ]);
    }


    public function changePassword(Request $request)
    {
        $request->validate([
            'old_password' => 'required',
            'new_password' => 'required|string|min:8|confirmed',
        ]);

        $user = $request->user();

        // Check old password
        if (!Hash::check($request->old_password, $user->password)) {
            return response()->json([
                'message' => 'Old password is incorrect.',
            ], 422);
        }

        // Update to new password
        $user->password = Hash::make($request->new_password);
        $user->save();

        return response()->json([
            'message' => 'Password changed successfully.',
        ]);
    }

    public function forgotPassword(Request $request) {
        $request->validate([
            "member_email" => "required|email|exists:members,member_email"
        ]);

        $member = Member::where("member_email", $request->member_email)->first();
        $password = rand(10000000, 99999999);
        $member->update([
            "password" => bcrypt($password)
        ]);

        try {
            $member->notify(new SendNewPasswordNotification($password));
        } catch (\Throwable $th) {
            //throw $th;
        }


        return response()->json([
            "message" => "Password reset successfully",
            "password" => $member,
            "password_string" => $password
        ], 200);
    }


    public function verifyToken($token) {


        $tokenParts = explode('.', $token);
        $header = json_decode(base64_decode($tokenParts[0]), true);
        $kid = $header['kid'];

        // Step 2: Fetch Google's public keys
        $keys = Http::get('https://www.googleapis.com/oauth2/v3/certs')->json();
        $decoded = null;

        try {
            $decoded = JWT::decode($token, JWK::parseKeySet($keys));
        } catch (\Exception $e) {
            return response()->json(['error' => 'Invalid token'], 401);
        }
        $membercheck = Member::where("member_email", $decoded->email)->first();
        if ($membercheck) {
            $token = JWTAuth::fromUser($membercheck);
            return response()->json([
                'access_token' => $token,
                'token_type' => 'bearer',
                'member' => $membercheck,
                'expires_in' => auth('member')->factory()->getTTL() * 60
            ]);
        }

        $member = Member::create([
            "member_email" => $decoded->email,
            "member_name" =>  $decoded->given_name." ".$decoded->family_name,
            "member_id" => "MEM".rand(100000, 999999)
        ]);
        $now = Carbon::now();
        $member->update([
            "next_subscription_date" => $now->addHour()
        ]);
        $token = JWTAuth::fromUser($member);
        return response()->json([
            'access_token' => $token,
            'token_type' => 'bearer',
            'member' => $member,
            'expires_in' => auth('member')->factory()->getTTL() * 60
        ]);
    }

    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        //
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        //
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $request->validate([
            "member_name" => "required|string",
            "member_email" => "required|email|unique:members,member_email",
            "password" => "required|string",
        ]);

        $member = Member::create([
            "member_name" => $request->member_name,
            "member_email" => $request->member_email,
            "member_id" => rand(100000,999999),
            "password" => bcrypt($request->password)
        ]);
        $now = Carbon::now();
        $member->update([
            "next_subscription_date" => $now->addHour()
        ]);

        $token = auth('member')->login($member);
        return response()->json([
            "message" => "Success",
            "data" => $member,
            "token" => $token
        ], 200);
    }

    /**
     * Display the specified resource.
     */
    public function show(member $member)
    {
        //
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(member $member)
    {
        //
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, member $member)
    {
        //
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(member $member)
    {
        //
    }

    public function verifyTransaction($referenceCode) {
        $powerbill = BookingPayment::where("payment_id", $referenceCode)->first();
        $booking = Booking::where("booking_id", $powerbill->booking_id)->first();
        if ($powerbill == null) {
            return response()->json(["message" => "Invalid reference"], 400);
        }


        if ($powerbill->status  == BookingPayment::PAID) {
            return response()->json([
                "message" => "Successful",
                "data" => $powerbill
            ], 200);

        }
        $response = (new SafeHavenService())->verifyTransaction($referenceCode);
        // return $response;
        if ($response == null) {
            return response()->json(["message" => "Invalid provider response"], 400);
        }

        if ($response["statusCode"] != 200) {
            return response()->json(["message" => $response["message"]], 400);
        }

        if ($response["data"]["status"] != 'Paid') {
            return response()->json(["message" => "Not Paid"], 400);
        }

        $powerbill->update([
            "status" => BookingPayment::PAID,
            "payment_response" => json_encode($response["data"])
        ]);
        // return $response["data"];

        $booking->update([
            "booking_status" => Booking::OPEN
        ]);


        return response()->json([
            "message" => "Successful",
            "data" => $powerbill
        ], 200);

    }
}
