<?php
namespace App\Notifications;

use Illuminate\Bus\Queueable;
use Illuminate\Notifications\Notification;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Notifications\Messages\MailMessage;
use Carbon\Carbon;

class SubscriptionActivated extends Notification implements ShouldQueue
{
    use Queueable;

    protected $nextPaymentDate;

    /**
     * Create a new notification instance.
     *
     * @param  \Carbon\Carbon|string  $nextPaymentDate
     */
    public function __construct($nextPaymentDate)
    {
        $this->nextPaymentDate = $nextPaymentDate instanceof Carbon
            ? $nextPaymentDate
            : Carbon::parse($nextPaymentDate);
    }

    /**
     * Get the notification's delivery channels.
     */
    public function via($notifiable)
    {
        return ['mail', 'database']; // Add SMS, Slack, etc., if needed
    }

    /**
     * Get the mail representation of the notification.
     */
    public function toMail($notifiable)
    {
        return (new MailMessage)
            ->subject('Your Subscription is Now Active')
            ->greeting('Hi ' . $notifiable->name . ',')
            ->line('🎉 Your subscription has been successfully activated.')
            ->line('📅 It will end on: **' . $this->nextPaymentDate->format('F j, Y') . '**.')
            ->line('Thank you for subscribing!');
    }

    /**
     * Get the array representation of the notification for database storage.
     */
    public function toArray($notifiable)
    {
        return [
            'title' => 'Subscription Activated',
            'message' => 'Your subscription is now active and will end on ' . $this->nextPaymentDate->toFormattedDateString() . '.',
            'next_payment_date' => $this->nextPaymentDate,
        ];
    }
}
